use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Getopt::Long;
use feature "state";

my $gnuplot = 'C:\Program Files (x86)\gnuplot\bin\gnuplot';


######################################################################
# 
# Script to generate pecentiles like Tech report
#
######################################################################
my $infile;
my $outfile;
my @infiles;
my @outfiles;
my $debug;
my $common;
my @gpus;
my $game;
my @extras;

my $fpsmode = 1;

my $result = GetOptions ( "infile=s"   => \@infiles,                      # input file  
			  "outf=s"     => \@outfiles,
			  "gpu=s"      => \@gpus,                         # GPU name
			  "extras=s"   => \@extras,                       # extra run info (eg res)
			  "fps"        => \$fpsmode,                      # Show FPS instead of percentiles
			  "game=s"     => \$game,                         # game
			  "common=s"   => \$common,
			  "debug"      => \$debug);                       # Turn on messages


my $of;
if ($common) { 
    $of = "$common";
}

# generate header of plot file
open PLOT, "> plot3";
plot_header($of, $game);

# Read the raw data file
my $ofi = 0;
my $plots = 0;

foreach my $infile (@infiles) {
    my @data;

    if (exists $outfiles[$ofi]) {
	$outfile = $outfiles[$ofi];
    }
    
    my $gpu;
    if (exists $gpus[$ofi]) {
	$gpu = $gpus[$ofi];
    }

    my $extra = "";
    if (exists $extras[$ofi]) {
	$extra = $extras[$ofi];
    }

    $ofi++;

    my $rc = readdata($infile, \@data);
    next if ($rc == 1);

    open OF, "> $outfile";
    printf ("Percentile Calcs on $gpu -> $outfile\n");

    parse_datafile(\@data);
    
    close OF;

    add_plot($plots, $outfile, $gpu. $extra);
    $plots++;

}

close PLOT;

#Run Gnuplot
my $rc2 = system ($gnuplot, "plot3");
if ($rc2) {
    printf ("Error running GNUPLOT\n");
}


#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
#
# Process bars
#
#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
sub parse_datafile {
    my ($ra_data) = @_;
    
    my $a_ref;
    my @d;
    while ($a_ref = shift @$ra_data) {
	my $t = @$a_ref[1];
	push @d, $t;
    }
    # numerical sort 
    my @sorted = sort { $a <=> $b } @d;
    my $cnt = @sorted;

    #find the max frame time of each percentile
    my $per;
    my $i=0;
    for ($per=0; $per < 100; $per++) {
	my $max = 0;
	for ($i=int(($per*$cnt)/100); $i<int((($per+1)*$cnt)/100); $i++) {
	    if ($sorted[$i] > $max) {
		$max = $sorted[$i];
	    }

	}

	# add option to display min FPS
	if ($fpsmode) {
	    $max = (1.0/$max)*1000;
	}
	printf OF ("%d,    $max\n", $per+1);
    }
}

#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
#
# Read the entire datafile
#
#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------
sub readdata {
    my ($filename, $ra_data) = @_;
    
#    printf ("Processing $filename....\n");
    open FILE, $filename or return (1);

    my @file = <FILE>;
    close FILE;
    
    shift @file; # skip header
    
    # Load in file (tab or comma seperated)
    @$ra_data = map { chomp; my @d = split /\s*[\,\t]\s*/; \@d; } @file;
}




# Print the fancy gnuplot graph
sub plot_header {
	    
    my ($outf, $game) = @_;


    # Auto axis?
    my $xauto = "set autoscale x";
    my $xrange = "set xrange [0:100]";
    
    my $yauto = "set autoscale y";
    my $yrange = "set yrange [0:]";

    # Insert plot setup into file
    print PLOT << "EOT";
set term pngcairo size 1920, 1080
set xlabel "Percentile"
set style fill transparent solid 0.5 noborder
$xauto
$xrange
$yauto
$yrange
set output '$outf.png'
set title '$game' font "Sans,40"
set style line 1 lt 1 lw 2 pt 3 linecolor rgb "red" 
set style line 2 lt 1 lw 2 pt 3 linecolor rgb "brown"
set style line 3 lt 1 lw 2 pt 3 linecolor rgb "yellow"
set style line 4 lt 1 lw 2 pt 3 linecolor rgb "pink"
set style line 5 lt 1 lw 2 pt 3 linecolor rgb "#336600"
set style line 6 lt 1 lw 2 pt 3 linecolor rgb "black"
set style line 7 lt 1 lw 2 pt 3 linecolor rgb "blue"
set style line 8 lt 1 lw 2 pt 3 linecolor rgb "green"

set grid ytics lt 0 lw 1 lc rgb "#880000"
set grid xtics lt 0 lw 1 lc rgb "#880000"


EOT
    

    if ($fpsmode) {
	printf PLOT ("set ylabel \"Min FPS\"\n");
    } else {
	printf PLOT ("set ylabel \"Frametime (ms)\"\n");
    }
}


# Print the fancy gnuplot graph
sub add_plot {
	    
    my ($cnt, $datafile, $gpu, $extra) = @_;

    # Pick a line type
    state $amd = 0;
    state $nv = 0;

    my $hwlt = 1;

    unless ($gpu) { $gpu = "Default"};

    if ($cnt == 0) {
	printf PLOT ("plot ");
    } else {
	printf PLOT ("\, \\\n     ");
    }
    
    # Pick NV or ATI colors
    if ($gpu =~ /^gtx/i) {
	$hwlt = 5 + $nv++;
    } elsif ($gpu =~ /^gk/i) {
	$hwlt = 5 + $nv++;
    } else {
	$hwlt = 1 + $amd++;
    }
    
    # Add HW plot
    unless ($extra) {
	$extra = "";
    }
    printf PLOT ("\'$datafile\' using 1:2 ls $hwlt ti \"$gpu $extra\" with lines");


}




